<?php 
require_once 'includes/config.php';
requireLogin();

if (!isset($_GET['order_id'])) {
    header('Location: packages.php');
    exit();
}

$order_id = $_GET['order_id'];
$user_id = $_SESSION['user_id'];

// Get order details
$stmt = $pdo->prepare("SELECT o.*, p.name as package_name FROM orders o JOIN packages p ON o.package_id = p.id WHERE o.id = ? AND o.user_id = ?");
$stmt->execute([$order_id, $user_id]);
$order = $stmt->fetch();

if (!$order) {
    header('Location: packages.php');
    exit();
}

if ($order['status'] !== 'pending') {
    header('Location: order-details.php?id=' . $order_id);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $payment_method = $order['payment_method'];
    $transaction_id = trim($_POST['transaction_id']);
    
    // Validate transaction ID
    if (empty($transaction_id) && $payment_method !== 'balance') {
        $error = "Transaction ID is required";
    } else {
        // Process payment
        if ($payment_method === 'balance') {
            // Check if user has sufficient balance
            $balance = getUserBalance($user_id);
            if ($balance >= $order['amount']) {
                // Deduct from balance
                updateUserBalance($user_id, -$order['amount']);
                
                // Update order status
                $stmt = $pdo->prepare("UPDATE orders SET status = 'completed', transaction_id = 'BALANCE-' . ? WHERE id = ?");
                $stmt->execute([time(), $order_id]);
                
                // Create subscription
                $start_date = date('Y-m-d');
                $end_date = date('Y-m-d', strtotime('+1 month'));
                
                $stmt = $pdo->prepare("INSERT INTO subscriptions (user_id, package_id, order_id, start_date, end_date) VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$user_id, $order['package_id'], $order_id, $start_date, $end_date]);
                
                $_SESSION['success_message'] = "Payment successful! Your hosting account is now active.";
                header('Location: order-complete.php?id=' . $order_id);
                exit();
            } else {
                $error = "Insufficient account balance";
            }
        } else {
            // For other payment methods, we'll just mark as completed (in real app, verify payment first)
            $stmt = $pdo->prepare("UPDATE orders SET status = 'completed', transaction_id = ? WHERE id = ?");
            $stmt->execute([$transaction_id, $order_id]);
            
            // Create subscription
            $start_date = date('Y-m-d');
            $end_date = date('Y-m-d', strtotime('+1 month'));
            
            $stmt = $pdo->prepare("INSERT INTO subscriptions (user_id, package_id, order_id, start_date, end_date) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$user_id, $order['package_id'], $order_id, $start_date, $end_date]);
            
            $_SESSION['success_message'] = "Payment successful! Your hosting account is now active.";
            header('Location: order-complete.php?id=' . $order_id);
            exit();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment - Web Hosting Bangladesh</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body>
    <header>
        <div class="container">
            <div class="logo">
                <a href="index.php"><img src="assets/images/logo.png" alt="Hosting BD Logo"></a>
            </div>
            <nav>
                <ul>
                    <li><a href="index.php">Home</a></li>
                    <li><a href="packages.php">Hosting Packages</a></li>
                    <li><a href="#features">Features</a></li>
                    <li><a href="#support">Support</a></li>
                    <?php if (isLoggedIn()): ?>
                        <li><a href="dashboard.php">Dashboard</a></li>
                        <li><a href="logout.php">Logout</a></li>
                    <?php else: ?>
                        <li><a href="login.php">Login</a></li>
                        <li><a href="register.php">Register</a></li>
                    <?php endif; ?>
                </ul>
            </nav>
        </div>
    </header>

    <section class="payment-page">
        <div class="container">
            <div class="payment-steps">
                <div class="step">
                    <span>1</span>
                    <p>Choose Package</p>
                </div>
                <div class="step">
                    <span>2</span>
                    <p>Configure</p>
                </div>
                <div class="step active">
                    <span>3</span>
                    <p>Payment</p>
                </div>
                <div class="step">
                    <span>4</span>
                    <p>Complete</p>
                </div>
            </div>
            
            <div class="payment-grid">
                <div class="payment-summary">
                    <h3>Order Summary</h3>
                    <div class="summary-card">
                        <h4>Order #<?php echo $order_id; ?></h4>
                        <div class="summary-details">
                            <div class="detail-row">
                                <span>Package:</span>
                                <span><?php echo htmlspecialchars($order['package_name']); ?></span>
                            </div>
                            <div class="detail-row">
                                <span>Billing Cycle:</span>
                                <span>Monthly</span>
                            </div>
                            <div class="detail-row">
                                <span>Payment Method:</span>
                                <span>
                                    <?php 
                                    $method_names = [
                                        'bkash' => 'bKash',
                                        'nagad' => 'Nagad',
                                        'rocket' => 'Rocket',
                                        'card' => 'Credit/Debit Card',
                                        'balance' => 'Account Balance'
                                    ];
                                    echo $method_names[$order['payment_method']]; 
                                    ?>
                                </span>
                            </div>
                            <div class="detail-row total">
                                <span>Total Amount:</span>
                                <span>৳<?php echo number_format($order['amount'], 2); ?></span>
                            </div>
                        </div>
                    </div>
                    
                    <?php if ($order['payment_method'] !== 'balance'): ?>
                    <div class="payment-instructions">
                        <h4>Payment Instructions</h4>
                        <div class="instructions">
                            <?php if ($order['payment_method'] === 'bkash'): ?>
                                <p>1. Go to your bKash Mobile Menu by dialing *247#</p>
                                <p>2. Choose "Payment" option</p>
                                <p>3. Enter our bKash Account Number: <strong>01234567890</strong></p>
                                <p>4. Enter the amount: <strong>৳<?php echo number_format($order['amount'], 2); ?></strong></p>
                                <p>5. Enter your transaction ID in the form</p>
                            <?php elseif ($order['payment_method'] === 'nagad'): ?>
                                <p>1. Go to your Nagad Mobile Menu</p>
                                <p>2. Choose "Send Money" option</p>
                                <p>3. Enter our Nagad Account Number: <strong>01234567890</strong></p>
                                <p>4. Enter the amount: <strong>৳<?php echo number_format($order['amount'], 2); ?></strong></p>
                                <p>5. Enter your transaction ID in the form</p>
                            <?php elseif ($order['payment_method'] === 'rocket'): ?>
                                <p>1. Go to your Rocket Mobile Menu by dialing *322#</p>
                                <p>2. Choose "Payment" option</p>
                                <p>3. Enter our Rocket Account Number: <strong>01234567890</strong></p>
                                <p>4. Enter the amount: <strong>৳<?php echo number_format($order['amount'], 2); ?></strong></p>
                                <p>5. Enter your transaction ID in the form</p>
                            <?php else: ?>
                                <p>You will be redirected to our secure payment gateway to complete your payment with your credit/debit card.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div class="payment-form">
                    <h3>Complete Payment</h3>
                    
                    <?php if (isset($error)): ?>
                        <div class="alert alert-danger">
                            <p><?php echo $error; ?></p>
                        </div>
                    <?php endif; ?>
                    
                    <form action="payment.php?order_id=<?php echo $order_id; ?>" method="POST">
                        <?php if ($order['payment_method'] === 'balance'): ?>
                            <div class="balance-payment">
                                <p>You're paying with your account balance.</p>
                                <p>Current Balance: <strong>৳<?php echo number_format(getUserBalance($user_id), 2); ?></strong></p>
                                <p>Amount to Pay: <strong>৳<?php echo number_format($order['amount'], 2); ?></strong></p>
                                
                                <?php if (getUserBalance($user_id) >= $order['amount']): ?>
                                    <p>After payment, your new balance will be: <strong>৳<?php echo number_format(getUserBalance($user_id) - $order['amount'], 2); ?></strong></p>
                                <?php else: ?>
                                    <p class="text-danger">You don't have sufficient balance to complete this payment.</p>
                                <?php endif; ?>
                            </div>
                        <?php else: ?>
                            <div class="form-group">
                                <label for="transaction_id">Transaction ID</label>
                                <input type="text" id="transaction_id" name="transaction_id" required placeholder="Enter your transaction ID">
                                <small>This is the transaction ID you received after making the payment</small>
                            </div>
                        <?php endif; ?>
                        
                        <button type="submit" class="btn btn-block" <?php echo ($order['payment_method'] === 'balance' && getUserBalance($user_id) < $order['amount']) ? 'disabled' : ''; ?>>
                            Complete Payment
                        </button>
                        
                        <div class="payment-security">
                            <i class="fas fa-lock"></i>
                            <span>Secure Payment</span>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </section>

    <footer>
        <div class="container">
            <div class="footer-grid">
                <div class="footer-col">
                    <h3>Hosting BD</h3>
                    <p>Providing affordable and reliable web hosting solutions in Bangladesh since 2023.</p>
                </div>
                <div class="footer-col">
                    <h3>Quick Links</h3>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="packages.php">Hosting Packages</a></li>
                        <li><a href="#features">Features</a></li>
                        <li><a href="#support">Support</a></li>
                    </ul>
                </div>
                <div class="footer-col">
                    <h3>Contact Us</h3>
                    <ul>
                        <li><i class="fas fa-phone"></i> +880 1XXX-XXXXXX</li>
                        <li><i class="fas fa-envelope"></i> support@hostingbd.com</li>
                        <li><i class="fas fa-map-marker-alt"></i> Dhaka, Bangladesh</li>
                    </ul>
                </div>
            </div>
            <div class="copyright">
                <p>&copy; <?php echo date('Y'); ?> Hosting BD. All Rights Reserved.</p>
            </div>
        </div>
    </footer>

    <script src="assets/js/script.js"></script>
</body>
</html>